# Installation Guide - MySQL + Apache Setup

This guide will help you install and configure the Stock Market Predictor application with MySQL database and Apache web server.

## Prerequisites

- Ubuntu/Debian Linux server
- Root or sudo access
- Apache 2.4+
- MySQL 8.0+
- Node.js 16+
- npm or yarn

## Step 1: Install Required Software

```bash
# Update system packages
sudo apt update && sudo apt upgrade -y

# Install Apache
sudo apt install apache2 -y

# Install MySQL
sudo apt install mysql-server -y

# Install Node.js (if not already installed)
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install nodejs -y

# Verify installations
apache2 -v
mysql --version
node --version
npm --version
```

## Step 2: Configure MySQL Database

### 2.1 Secure MySQL Installation

```bash
sudo mysql_secure_installation
```

Follow the prompts to set a root password and secure your MySQL installation.

### 2.2 Create Database and User

```bash
# Login to MySQL as root
sudo mysql -u root -p

# Or run the setup script directly
sudo mysql -u root -p < setup-database.sql
```

If running manually:

```sql
-- Create database
CREATE DATABASE stock_predictor CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Create user (change the password!)
CREATE USER 'stock_predictor_user'@'localhost' IDENTIFIED BY 'YOUR_SECURE_PASSWORD';

-- Grant privileges
GRANT ALL PRIVILEGES ON stock_predictor.* TO 'stock_predictor_user'@'localhost';
FLUSH PRIVILEGES;

-- Exit
EXIT;
```

## Step 3: Configure Application

### 3.1 Install Node.js Dependencies

```bash
cd /home/antoine/server_files/traider.antoinepelloux.eu
npm install
```

### 3.2 Configure Environment Variables

```bash
# Copy the example file
cp .env.example .env

# Edit the .env file
nano .env
```

Update these values:

```env
PORT=3000
NODE_ENV=production

# Your Claude AI API key
ANTHROPIC_API_KEY=sk-ant-your-actual-key-here

# MySQL credentials (match what you created in Step 2.2)
DB_HOST=localhost
DB_USER=stock_predictor_user
DB_PASSWORD=YOUR_SECURE_PASSWORD
DB_NAME=stock_predictor

ENABLE_SCHEDULER=true
```

### 3.3 Test the Application

```bash
# Test if the app starts correctly
npm start
```

You should see:
```
Connected to MySQL database
Database tables initialized successfully
Stock Market Predictor - Server Started
Server running on: http://localhost:3000
```

Press Ctrl+C to stop. If there are any errors, check your MySQL credentials.

## Step 4: Configure Apache Web Server

### 4.1 Enable Required Apache Modules

```bash
sudo a2enmod proxy
sudo a2enmod proxy_http
sudo a2enmod proxy_wstunnel
sudo a2enmod rewrite
sudo a2enmod headers
sudo a2enmod ssl
sudo a2enmod deflate
sudo a2enmod expires

# Restart Apache
sudo systemctl restart apache2
```

### 4.2 Configure Virtual Host

```bash
# Copy the Apache configuration
sudo cp apache-config/traider.antoinepelloux.eu.conf /etc/apache2/sites-available/

# Edit if needed
sudo nano /etc/apache2/sites-available/traider.antoinepelloux.eu.conf

# Enable the site
sudo a2ensite traider.antoinepelloux.eu.conf

# Disable default site (optional)
sudo a2dissite 000-default.conf

# Test Apache configuration
sudo apache2ctl configtest

# Reload Apache
sudo systemctl reload apache2
```

## Step 5: Set Up Node.js as a System Service

### 5.1 Create Systemd Service

```bash
# Copy the service file
sudo cp apache-config/stock-predictor.service /etc/systemd/system/

# Reload systemd
sudo systemctl daemon-reload

# Enable the service to start on boot
sudo systemctl enable stock-predictor

# Start the service
sudo systemctl start stock-predictor

# Check status
sudo systemctl status stock-predictor
```

### 5.2 Manage the Service

```bash
# Start service
sudo systemctl start stock-predictor

# Stop service
sudo systemctl stop stock-predictor

# Restart service
sudo systemctl restart stock-predictor

# View logs
sudo journalctl -u stock-predictor -f

# View last 100 lines
sudo journalctl -u stock-predictor -n 100
```

## Step 6: Configure Firewall

```bash
# Allow Apache traffic
sudo ufw allow 'Apache Full'

# Or allow specific ports
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp

# Enable firewall (if not already enabled)
sudo ufw enable

# Check status
sudo ufw status
```

## Step 7: Set Up SSL/HTTPS (Optional but Recommended)

### Using Let's Encrypt (Certbot)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-apache -y

# Obtain and install certificate
sudo certbot --apache -d traider.antoinepelloux.eu

# Test auto-renewal
sudo certbot renew --dry-run
```

Certbot will automatically configure SSL in your Apache virtual host.

## Step 8: Verify Installation

### 8.1 Check Services

```bash
# Check MySQL
sudo systemctl status mysql

# Check Apache
sudo systemctl status apache2

# Check Node.js app
sudo systemctl status stock-predictor
```

### 8.2 Test the Website

Open your browser and navigate to:
- `http://traider.antoinepelloux.eu`
- Or `https://traider.antoinepelloux.eu` if you set up SSL

You should see the Stock Market Predictor dashboard.

### 8.3 Test API Endpoints

```bash
# Test health endpoint
curl http://traider.antoinepelloux.eu/api/health

# Should return: {"status":"ok","timestamp":"..."}

# Test market data (may take a moment)
curl http://traider.antoinepelloux.eu/api/market-data
```

## Step 9: Run First Prediction

### Via Web Interface

1. Open `http://traider.antoinepelloux.eu`
2. Click "Run New Prediction" button
3. Wait for the process to complete (1-2 minutes)

### Via Command Line

```bash
curl -X POST http://localhost:3000/api/predict/run
```

## Troubleshooting

### Application won't start

```bash
# Check logs
sudo journalctl -u stock-predictor -n 50

# Common issues:
# 1. MySQL connection - check credentials in .env
# 2. Missing dependencies - run: npm install
# 3. Port already in use - change PORT in .env
```

### Apache not proxying correctly

```bash
# Verify proxy modules are enabled
apache2ctl -M | grep proxy

# Check Apache error log
sudo tail -f /var/log/apache2/traider-error.log

# Check Apache access log
sudo tail -f /var/log/apache2/traider-access.log
```

### Database connection errors

```bash
# Test MySQL connection
mysql -u stock_predictor_user -p stock_predictor

# If it fails, recreate the user
sudo mysql -u root -p
> DROP USER 'stock_predictor_user'@'localhost';
> CREATE USER 'stock_predictor_user'@'localhost' IDENTIFIED BY 'password';
> GRANT ALL PRIVILEGES ON stock_predictor.* TO 'stock_predictor_user'@'localhost';
> FLUSH PRIVILEGES;
```

### Scheduler not running

Check the .env file:
```bash
ENABLE_SCHEDULER=true
```

Restart the service:
```bash
sudo systemctl restart stock-predictor
```

## Maintenance

### Update the Application

```bash
# Stop the service
sudo systemctl stop stock-predictor

# Pull latest changes (if using git)
git pull

# Install dependencies
npm install

# Restart service
sudo systemctl start stock-predictor
```

### Database Backup

```bash
# Backup database
mysqldump -u stock_predictor_user -p stock_predictor > backup_$(date +%Y%m%d).sql

# Restore database
mysql -u stock_predictor_user -p stock_predictor < backup_20231201.sql
```

### View Application Logs

```bash
# Real-time logs
sudo journalctl -u stock-predictor -f

# Logs from today
sudo journalctl -u stock-predictor --since today

# Logs with specific time range
sudo journalctl -u stock-predictor --since "2023-12-01 00:00:00" --until "2023-12-01 23:59:59"
```

## Performance Optimization

### 1. Enable Apache caching

Already configured in the virtual host config file.

### 2. MySQL optimization

Edit `/etc/mysql/mysql.conf.d/mysqld.cnf`:

```ini
[mysqld]
innodb_buffer_pool_size = 256M
max_connections = 50
query_cache_size = 16M
query_cache_type = 1
```

Restart MySQL:
```bash
sudo systemctl restart mysql
```

### 3. Node.js process management

For production, consider using PM2 instead of systemd:

```bash
npm install -g pm2
pm2 start src/app.js --name stock-predictor
pm2 save
pm2 startup
```

## Security Checklist

- ✅ MySQL root password set
- ✅ Application database user has limited privileges
- ✅ .env file contains secure passwords
- ✅ .env file permissions: `chmod 600 .env`
- ✅ Firewall enabled and configured
- ✅ SSL/HTTPS enabled
- ✅ Apache security headers configured
- ✅ Regular backups scheduled

## Support

For issues or questions, check:
- Application logs: `sudo journalctl -u stock-predictor -f`
- Apache logs: `/var/log/apache2/`
- MySQL logs: `/var/log/mysql/`

## Next Steps

1. Set up automated backups
2. Configure monitoring (e.g., Uptime Robot, Prometheus)
3. Set up email notifications for predictions
4. Review and optimize cron schedule times
5. Consider adding authentication for API endpoints

---

Your Stock Market Predictor is now installed and running with MySQL and Apache!
