# European Stock Market Predictor - PHP Version

A pure PHP implementation of the stock market prediction application. No Node.js required - runs entirely on Apache with PHP.

## Features

- **Pure PHP Backend**: No Node.js, no npm dependencies
- **Apache Integration**: Runs directly on Apache web server
- **MySQL Database**: Stores predictions and historical data
- **Claude AI Integration**: Uses Anthropic's Claude for predictions
- **Cron-based Scheduling**: Daily automated predictions and scoring
- **Same Frontend**: Beautiful, responsive web interface

## Architecture

```
├── api/                         # PHP API endpoints
│   ├── health.php              # Health check
│   ├── market-data.php         # Yahoo Finance data
│   ├── news.php                # Google News
│   ├── predictions.php         # Latest predictions
│   ├── scores.php              # Historical scores
│   ├── run-prediction.php      # Manual prediction trigger
│   └── run-scoring.php         # Manual scoring trigger
├── config/
│   ├── config.php              # Configuration loader
│   └── database.php            # MySQL PDO connection
├── services/
│   ├── YahooFinanceService.php # Market data fetching
│   ├── GoogleNewsService.php   # News aggregation
│   ├── ClaudeAIService.php     # AI predictions
│   └── ScoringService.php      # Database operations & scoring
├── cron/
│   ├── daily-prediction.php    # 6 PM daily prediction
│   └── daily-scoring.php       # 9 AM daily scoring
└── public/
    ├── index.html              # Frontend
    ├── styles.css
    └── app.js
```

## Requirements

- **PHP 7.4+** with extensions:
  - curl
  - pdo_mysql
  - xml
  - mbstring
- **Apache 2.4+**
- **MySQL 8.0+**
- **Anthropic API Key**

## Quick Installation

### 1. Install Prerequisites

```bash
sudo apt update
sudo apt install -y apache2 php php-curl php-mysql php-xml php-mbstring mysql-server
```

### 2. Set Up MySQL Database

```bash
sudo mysql -u root -p < setup-database.sql
```

Don't forget to edit `setup-database.sql` first and change the password!

### 3. Configure Environment

```bash
cp .env.example .env
nano .env
```

Update these values:
```env
ANTHROPIC_API_KEY=your_actual_api_key
DB_HOST=localhost
DB_USER=stock_predictor_user
DB_PASSWORD=your_secure_password
DB_NAME=stock_predictor
```

### 4. Run Installation Script

```bash
chmod +x install-php.sh
./install-php.sh
```

This will:
- Install Apache configuration
- Enable required modules
- Set file permissions
- Initialize database tables
- Set up cron jobs
- Reload Apache

### 5. Test the Installation

```bash
# Test health endpoint
curl http://traider.antoinepelloux.eu/api/health

# Should return: {"status":"ok","timestamp":"..."}
```

## API Endpoints

### GET Endpoints

- `GET /api/health` - Health check
- `GET /api/market-data` - Current European market data
- `GET /api/news` - Recent financial news
- `GET /api/predictions` - Latest predictions with scores
- `GET /api/scores?limit=30` - Historical accuracy scores

### POST Endpoints

- `POST /api/run-prediction` - Manually trigger prediction
- `POST /api/run-scoring` - Manually score predictions

## Cron Jobs

The installation script automatically sets up two cron jobs:

**Daily Prediction** (6:00 PM CET)
```bash
0 18 * * * /usr/bin/php /path/to/cron/daily-prediction.php >> /var/log/stock-predictor-prediction.log 2>&1
```

**Daily Scoring** (9:00 AM CET)
```bash
0 9 * * * /usr/bin/php /path/to/cron/daily-scoring.php >> /var/log/stock-predictor-scoring.log 2>&1
```

View logs:
```bash
tail -f /var/log/stock-predictor-prediction.log
tail -f /var/log/stock-predictor-scoring.log
```

## Manual Testing

### Test Prediction Generation

```bash
php cron/daily-prediction.php
```

### Test Scoring

```bash
php cron/daily-scoring.php
```

### Test Individual Services

```bash
# Test Yahoo Finance
php -r "
require_once 'services/YahooFinanceService.php';
\$service = new YahooFinanceService();
\$data = \$service->fetchMarketData('^FTSE');
print_r(\$data);
"

# Test Google News
php -r "
require_once 'services/GoogleNewsService.php';
\$service = new GoogleNewsService();
\$news = \$service->fetchFinanceNews('FTSE 100', 5);
print_r(\$news);
"
```

## Troubleshooting

### Issue: "Class 'Database' not found"

**Solution**: Make sure you're requiring the config files:
```php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';
```

### Issue: MySQL connection failed

**Solution**: Check your .env credentials and verify MySQL is running:
```bash
sudo systemctl status mysql
mysql -u stock_predictor_user -p stock_predictor
```

### Issue: API returns 500 error

**Solution**: Check Apache error logs:
```bash
sudo tail -f /var/log/apache2/traider-error.log
```

### Issue: Cron jobs not running

**Solution**: Check crontab and logs:
```bash
crontab -l
tail -f /var/log/stock-predictor-prediction.log
```

## Advantages of PHP Version

✅ **No Node.js dependencies** - Pure PHP, easier to maintain
✅ **Native Apache integration** - No reverse proxy needed
✅ **Simpler deployment** - Just upload files and go
✅ **Standard cron jobs** - No systemd service required
✅ **Lower memory footprint** - PHP scripts run on-demand
✅ **Familiar stack** - LAMP/LEMP standard hosting

## File Permissions

The installation script sets these automatically, but for reference:

```bash
chmod 755 api/
chmod 644 api/*.php
chmod 755 cron/
chmod 755 cron/*.php
chmod 600 .env
chown -R www-data:www-data public/
```

## Security Notes

- The `.env` file is set to `chmod 600` and should never be web-accessible
- API endpoints validate request methods (POST endpoints only accept POST)
- All database queries use prepared statements (PDO)
- CORS headers are configured in `config/config.php`
- Apache security headers are set in the virtual host

## Performance

For high-traffic scenarios:

1. **Enable OPcache** in `php.ini`:
```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
```

2. **Use connection pooling** (already implemented via PDO)

3. **Cache API responses** using Apache mod_cache or Redis

## Monitoring

Monitor the application:

```bash
# Watch prediction logs in real-time
tail -f /var/log/stock-predictor-prediction.log

# Watch Apache errors
tail -f /var/log/apache2/traider-error.log

# Check cron job history
grep CRON /var/log/syslog | grep stock-predictor
```

## Support

For issues:
1. Check Apache error logs
2. Check cron job logs
3. Test PHP scripts manually
4. Verify MySQL connectivity
5. Ensure API keys are valid

---

**Note**: This is a complete rewrite of the application in PHP. No Node.js components remain. The frontend is identical but now communicates with PHP endpoints instead of a Node.js Express server.
