<?php
/**
 * Yahoo Finance Service
 * Fetches stock market data from Yahoo Finance API
 */

class YahooFinanceService {

    private $europeanIndices = [
        ['symbol' => '^FTSE', 'name' => 'FTSE 100', 'country' => 'UK'],
        ['symbol' => '^GDAXI', 'name' => 'DAX', 'country' => 'Germany'],
        ['symbol' => '^FCHI', 'name' => 'CAC 40', 'country' => 'France'],
        ['symbol' => '^IBEX', 'name' => 'IBEX 35', 'country' => 'Spain'],
        ['symbol' => '^AEX', 'name' => 'AEX', 'country' => 'Netherlands'],
        ['symbol' => '^SSMI', 'name' => 'SMI', 'country' => 'Switzerland'],
        ['symbol' => 'FTSEMIB.MI', 'name' => 'FTSE MIB', 'country' => 'Italy'],
        ['symbol' => '^OMXC25', 'name' => 'OMX Copenhagen 25', 'country' => 'Denmark'],
        ['symbol' => '^OMXHPI', 'name' => 'OMX Helsinki', 'country' => 'Finland'],
        ['symbol' => '^OMXSPI', 'name' => 'OMX Stockholm', 'country' => 'Sweden'],
        ['symbol' => '^OSEAX', 'name' => 'Oslo Stock Exchange', 'country' => 'Norway'],
        ['symbol' => '^ISEQ', 'name' => 'ISEQ Overall', 'country' => 'Ireland'],
        ['symbol' => '^BFX', 'name' => 'BEL 20', 'country' => 'Belgium'],
        ['symbol' => '^PSI20', 'name' => 'PSI 20', 'country' => 'Portugal'],
        ['symbol' => '^ATX', 'name' => 'ATX', 'country' => 'Austria']
    ];

    /**
     * Fetch market data for a single symbol
     */
    public function fetchMarketData($symbol) {
        try {
            $url = "https://query1.finance.yahoo.com/v8/finance/chart/" . urlencode($symbol);
            $url .= "?interval=1d&range=5d";

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200 || !$response) {
                error_log("Failed to fetch data for $symbol: HTTP $httpCode");
                return null;
            }

            $data = json_decode($response, true);

            if (!isset($data['chart']['result'][0])) {
                error_log("No data in response for $symbol");
                return null;
            }

            $result = $data['chart']['result'][0];
            $quote = $result['indicators']['quote'][0];
            $timestamps = $result['timestamp'];

            if (empty($timestamps)) {
                return null;
            }

            $lastIdx = count($timestamps) - 1;
            $prevIdx = $lastIdx - 1;

            $currentClose = $quote['close'][$lastIdx];
            $previousClose = $prevIdx >= 0 ? $quote['close'][$prevIdx] : $currentClose;
            $changePercent = $previousClose != 0 ? (($currentClose - $previousClose) / $previousClose) * 100 : 0;

            return [
                'symbol' => $symbol,
                'timestamp' => $timestamps[$lastIdx],
                'open' => $quote['open'][$lastIdx],
                'close' => $currentClose,
                'high' => $quote['high'][$lastIdx],
                'low' => $quote['low'][$lastIdx],
                'volume' => $quote['volume'][$lastIdx],
                'previousClose' => $previousClose,
                'changePercent' => $changePercent
            ];

        } catch (Exception $e) {
            error_log("Error fetching data for $symbol: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Fetch all European indices
     */
    public function fetchAllEuropeanIndices() {
        $results = [];

        foreach ($this->europeanIndices as $index) {
            $data = $this->fetchMarketData($index['symbol']);

            if ($data) {
                $data['name'] = $index['name'];
                $data['country'] = $index['country'];
                $results[] = $data;
            }

            // Small delay to avoid rate limiting
            usleep(500000); // 0.5 seconds
        }

        return $results;
    }

    /**
     * Get European indices list
     */
    public function getEuropeanIndices() {
        return $this->europeanIndices;
    }
}
