require('dotenv').config();
const express = require('express');
const cors = require('cors');
const path = require('path');
const apiRoutes = require('./routes/api');
const { initializeScheduler } = require('./services/scheduler');
const db = require('./config/database');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, '../public')));

// API Routes
app.use('/api', apiRoutes);

// Serve frontend
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, '../public/index.html'));
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Error:', err);
  res.status(500).json({
    success: false,
    error: 'Internal server error',
    message: err.message
  });
});

// Start server
app.listen(PORT, () => {
  console.log('\n===========================================');
  console.log('  Stock Market Predictor - Server Started');
  console.log('===========================================');
  console.log(`  Server running on: http://localhost:${PORT}`);
  console.log(`  Environment: ${process.env.NODE_ENV || 'development'}`);
  console.log(`  Database: MySQL (${process.env.DB_NAME || 'stock_predictor'})`);
  console.log('===========================================\n');

  // Initialize scheduler for daily tasks
  if (process.env.ENABLE_SCHEDULER !== 'false') {
    initializeScheduler();
  } else {
    console.log('Scheduler disabled. Set ENABLE_SCHEDULER=true to enable.');
  }
});

// Graceful shutdown
process.on('SIGINT', async () => {
  console.log('\nShutting down gracefully...');
  try {
    await db.close();
    console.log('Database connection closed');
    process.exit(0);
  } catch (err) {
    console.error('Error closing database:', err);
    process.exit(1);
  }
});

module.exports = app;
