const Parser = require('rss-parser');
const parser = new Parser();

/**
 * Fetch news articles from Google News RSS feed
 */
async function fetchFinanceNews(query = 'European stock market', maxArticles = 20) {
  try {
    const searchQuery = encodeURIComponent(query);
    const rssUrl = `https://news.google.com/rss/search?q=${searchQuery}&hl=en-US&gl=US&ceid=US:en`;

    const feed = await parser.parseURL(rssUrl);

    const articles = feed.items.slice(0, maxArticles).map(item => ({
      title: item.title,
      link: item.link,
      pubDate: item.pubDate,
      source: item.source?.name || 'Unknown',
      contentSnippet: item.contentSnippet || ''
    }));

    return articles;
  } catch (error) {
    console.error('Error fetching Google News:', error.message);
    return [];
  }
}

/**
 * Fetch news for multiple European stock markets
 */
async function fetchEuropeanMarketNews() {
  const queries = [
    'European stock market',
    'FTSE 100',
    'DAX Germany',
    'CAC 40 France',
    'European economy',
    'European Central Bank'
  ];

  const allArticles = [];

  for (const query of queries) {
    const articles = await fetchFinanceNews(query, 5);
    allArticles.push(...articles);
    // Small delay to avoid rate limiting
    await new Promise(resolve => setTimeout(resolve, 1000));
  }

  // Remove duplicates based on title
  const uniqueArticles = Array.from(
    new Map(allArticles.map(article => [article.title, article])).values()
  );

  return uniqueArticles;
}

/**
 * Format news articles into a text summary for Claude
 */
function formatNewsForClaude(articles) {
  if (articles.length === 0) {
    return "No recent news articles available.";
  }

  let formatted = "Recent Financial News:\n\n";
  articles.forEach((article, index) => {
    formatted += `${index + 1}. ${article.title}\n`;
    formatted += `   Source: ${article.source}\n`;
    formatted += `   Published: ${article.pubDate}\n`;
    if (article.contentSnippet) {
      formatted += `   ${article.contentSnippet}\n`;
    }
    formatted += '\n';
  });

  return formatted;
}

module.exports = {
  fetchFinanceNews,
  fetchEuropeanMarketNews,
  formatNewsForClaude
};
