const cron = require('node-cron');
const { fetchAllEuropeanIndices } = require('./yahooFinance');
const { fetchEuropeanMarketNews } = require('./googleNews');
const { generatePredictions } = require('./claudeAI');
const {
  saveMarketData,
  saveNewsArticles,
  savePredictions,
  scorePredictions
} = require('./scoring');

/**
 * Run daily prediction task
 */
async function runDailyPrediction() {
  console.log('\n=== Running Daily Prediction Task ===');
  console.log('Time:', new Date().toISOString());

  try {
    // 1. Fetch current market data
    console.log('Fetching market data...');
    const marketData = await fetchAllEuropeanIndices();
    console.log(`Fetched data for ${marketData.length} indices`);

    // 2. Fetch news articles
    console.log('Fetching news articles...');
    const newsArticles = await fetchEuropeanMarketNews();
    console.log(`Fetched ${newsArticles.length} news articles`);

    // 3. Generate predictions using Claude
    console.log('Generating predictions with Claude AI...');
    const predictions = await generatePredictions(marketData, newsArticles);
    console.log('Predictions generated successfully');

    // 4. Save everything to database
    const today = new Date().toISOString().split('T')[0];
    const tomorrow = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString().split('T')[0];

    console.log('Saving market data...');
    await saveMarketData(today, marketData);

    console.log('Saving news articles...');
    await saveNewsArticles(today, newsArticles);

    console.log('Saving predictions...');
    await savePredictions(today, tomorrow, predictions);

    console.log('=== Daily Prediction Task Completed ===\n');

    return {
      success: true,
      date: today,
      marketDataCount: marketData.length,
      newsCount: newsArticles.length,
      predictions: predictions
    };
  } catch (error) {
    console.error('Error in daily prediction task:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Run daily scoring task
 */
async function runDailyScoring() {
  console.log('\n=== Running Daily Scoring Task ===');
  console.log('Time:', new Date().toISOString());

  try {
    const today = new Date().toISOString().split('T')[0];
    const yesterday = new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString().split('T')[0];

    // 1. Fetch today's market data (for scoring yesterday's predictions)
    console.log('Fetching today\'s market data for scoring...');
    const marketData = await fetchAllEuropeanIndices();
    await saveMarketData(today, marketData);

    // 2. Score yesterday's predictions
    console.log('Scoring yesterday\'s predictions...');
    const scores = await scorePredictions(yesterday, today);

    console.log('=== Scoring Results ===');
    console.log(`Total Predictions: ${scores.total_predictions}`);
    console.log(`Correct: ${scores.correct_predictions}`);
    console.log(`Accuracy: ${scores.accuracy_percent.toFixed(2)}%`);
    console.log('=== Daily Scoring Task Completed ===\n');

    return scores;
  } catch (error) {
    console.error('Error in daily scoring task:', error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Initialize scheduled tasks
 */
function initializeScheduler() {
  console.log('Initializing scheduler...');

  // Run predictions every day at 6 PM (18:00) CET
  // This runs after European markets close
  cron.schedule('0 18 * * *', async () => {
    console.log('Scheduled prediction task triggered');
    await runDailyPrediction();
  }, {
    timezone: "Europe/Paris"
  });

  // Run scoring every day at 9 AM (09:00) CET
  // This runs after markets open and we have fresh data
  cron.schedule('0 9 * * *', async () => {
    console.log('Scheduled scoring task triggered');
    await runDailyScoring();
  }, {
    timezone: "Europe/Paris"
  });

  console.log('Scheduler initialized:');
  console.log('- Predictions: Daily at 18:00 CET (after market close)');
  console.log('- Scoring: Daily at 09:00 CET (after market open)');
}

module.exports = {
  initializeScheduler,
  runDailyPrediction,
  runDailyScoring
};
