# European Stock Market Predictor

An AI-powered web application that predicts European stock market movements using Claude AI, Yahoo Finance data, and Google News.

## Features

- **Daily Market Analysis**: Collects real-time data from 15 major European stock indices
- **AI Predictions**: Uses Claude AI to predict top 5 indices likely to rise and top 5 likely to fall
- **News Integration**: Aggregates relevant financial news from Google News
- **Self-Scoring**: Automatically compares predictions with actual market performance
- **Historical Tracking**: Maintains accuracy scores over time
- **Beautiful UI**: Modern, responsive web interface

## European Indices Tracked

- FTSE 100 (UK)
- DAX (Germany)
- CAC 40 (France)
- IBEX 35 (Spain)
- AEX (Netherlands)
- SMI (Switzerland)
- FTSE MIB (Italy)
- And more...

## Tech Stack

- **Backend**: Node.js + Express
- **Database**: MySQL 8.0+
- **Web Server**: Apache 2.4+ (reverse proxy to Node.js)
- **AI**: Claude AI (Anthropic API)
- **Data Sources**: Yahoo Finance API, Google News RSS
- **Frontend**: HTML, CSS, JavaScript
- **Scheduling**: node-cron
- **Process Management**: systemd service

## Installation

### Quick Start (Development)

For development/testing without Apache:

```bash
# Install dependencies
npm install

# Create environment file
cp .env.example .env

# Edit .env and configure your MySQL credentials and API key
nano .env

# Start the server
npm start
```

### Production Installation (MySQL + Apache)

For complete production setup with MySQL database and Apache web server, see the detailed installation guide:

**[📖 Full Installation Guide (INSTALL.md)](INSTALL.md)**

The installation guide covers:
- MySQL database setup
- Apache web server configuration
- SSL/HTTPS setup with Let's Encrypt
- Systemd service configuration
- Security hardening
- Troubleshooting

### Prerequisites

- Ubuntu/Debian Linux server
- Apache 2.4+
- MySQL 8.0+
- Node.js 16+
- Anthropic API key (get one at https://console.anthropic.com/)

## Usage

### Automatic Daily Operations

The system automatically runs two scheduled tasks:

- **6:00 PM CET**: Generates predictions for the next day
  - Fetches current market data
  - Collects latest news
  - Generates AI predictions
  - Saves to database

- **9:00 AM CET**: Scores previous day's predictions
  - Fetches updated market data
  - Compares predictions with actual results
  - Calculates accuracy
  - Updates dashboard

### Manual Operations

You can manually trigger operations via the web interface:

1. **Refresh Data**: Updates all data on the dashboard
2. **Run New Prediction**: Manually triggers a prediction run

Or via API:

```bash
# Run prediction
curl -X POST http://localhost:3000/api/predict/run

# Run scoring
curl -X POST http://localhost:3000/api/score/run
```

## API Endpoints

### GET Endpoints

- `GET /api/health` - Health check
- `GET /api/market-data` - Current European market data
- `GET /api/news` - Recent financial news
- `GET /api/predictions/latest` - Latest predictions with scores
- `GET /api/scores/history?limit=30` - Historical accuracy scores

### POST Endpoints

- `POST /api/predict/run` - Manually run prediction
- `POST /api/score/run` - Manually run scoring

## Project Structure

```
stock-market-predictor/
├── src/
│   ├── config/
│   │   └── database.js          # MySQL database setup
│   ├── services/
│   │   ├── yahooFinance.js      # Yahoo Finance API integration
│   │   ├── googleNews.js        # Google News RSS integration
│   │   ├── claudeAI.js          # Claude AI predictions
│   │   ├── scoring.js           # Prediction scoring logic
│   │   └── scheduler.js         # Cron job scheduler
│   ├── routes/
│   │   └── api.js               # Express API routes
│   └── app.js                   # Main Express application
├── public/
│   ├── index.html               # Frontend HTML
│   ├── styles.css               # Frontend CSS
│   ├── app.js                   # Frontend JavaScript
│   └── .htaccess                # Apache rewrite rules
├── apache-config/
│   ├── traider.antoinepelloux.eu.conf  # Apache virtual host
│   └── stock-predictor.service         # Systemd service file
├── package.json
├── .env                         # Environment variables (create from .env.example)
├── .env.example                 # Example environment configuration
├── setup-database.sql           # MySQL database setup script
├── INSTALL.md                   # Detailed installation guide
├── .gitignore
└── README.md
```

## Database Schema

The application uses MySQL with the following tables:

- **market_data**: Daily market prices and changes
- **news_articles**: Collected news articles
- **predictions**: AI-generated predictions
- **prediction_scores**: Individual prediction results
- **daily_scores**: Daily accuracy aggregates

## How It Works

### 1. Data Collection
- Fetches real-time data from Yahoo Finance for 15 European indices
- Collects latest financial news from Google News RSS feeds
- All data is timestamped and stored in the database

### 2. AI Prediction
- Claude AI analyzes market data and news
- Considers technical indicators, trends, and sentiment
- Generates top 5 UP and top 5 DOWN predictions with reasoning

### 3. Scoring
- Next day, fetches updated market data
- Compares each prediction (UP/DOWN) with actual movement
- Calculates accuracy percentage
- Stores results for historical analysis

### 4. Visualization
- Modern web interface displays all information
- Color-coded predictions (green for correct, red for incorrect)
- Historical accuracy charts
- Real-time market data

## Configuration Options

### Disable Automatic Scheduling

To disable the cron scheduler:

```env
ENABLE_SCHEDULER=false
```

### Change Port

```env
PORT=8080
```

### MySQL Connection

```env
DB_HOST=localhost
DB_USER=your_db_user
DB_PASSWORD=your_db_password
DB_NAME=stock_predictor
```

## Development

Run in development mode with auto-reload:

```bash
npm run dev
```

## Troubleshooting

### Issue: No predictions showing

**Solution**: Run a manual prediction via the "Run New Prediction" button or:
```bash
curl -X POST http://localhost:3000/api/predict/run
```

### Issue: API key error

**Solution**: Make sure your `.env` file contains a valid Anthropic API key:
```env
ANTHROPIC_API_KEY=sk-ant-...
```

### Issue: Database connection error

**Solution**: Check your MySQL credentials in `.env` file and ensure MySQL service is running:
```bash
sudo systemctl status mysql
```

### Issue: Rate limiting from Yahoo Finance

**Solution**: The app includes delays between requests. If issues persist, consider reducing the number of indices tracked.

## Important Notes

- **Not Financial Advice**: This is an experimental AI project for educational purposes only
- **Market Hours**: European markets are closed on weekends and holidays
- **API Costs**: Using Claude AI API incurs costs based on usage
- **Data Accuracy**: Yahoo Finance data is generally reliable but may have delays

## Future Enhancements

Possible improvements:

- Add more technical indicators (RSI, MACD, etc.)
- Implement backtesting with historical data
- Add email/SMS notifications
- Create mobile app
- Add sentiment analysis from Twitter/Reddit
- Support more markets (US, Asian markets)
- Implement machine learning model alongside Claude AI

## License

MIT License

## Disclaimer

This application is for educational and research purposes only. Stock market predictions are inherently uncertain. Do not use this tool as the sole basis for investment decisions. Always consult with a qualified financial advisor before making investment choices.

## Support

For issues, questions, or contributions, please open an issue on the project repository.
