<?php
#!/bin/bash

# Installation Script for Stock Market Predictor (PHP Version)
# This script sets up the PHP-based application with Apache

set -e

echo "========================================"
echo "Stock Market Predictor - PHP Installation"
echo "========================================"
echo ""

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

# Step 1: Check PHP version
echo "1. Checking PHP installation..."
if ! command -v php &> /dev/null; then
    echo "   ✗ PHP is not installed!"
    echo "   Installing PHP..."
    sudo apt update
    sudo apt install -y php php-curl php-mysql php-xml php-mbstring
fi

PHP_VERSION=$(php -v | head -n 1 | cut -d " " -f 2 | cut -d "." -f 1,2)
echo "   ✓ PHP $PHP_VERSION detected"

# Step 2: Enable required Apache modules
echo "2. Enabling Apache modules..."
sudo a2enmod rewrite headers deflate expires php || true
echo "   ✓ Modules enabled"

# Step 3: Copy Apache configuration
echo "3. Installing Apache configuration..."
sudo cp "$SCRIPT_DIR/apache-config/004-traider.antoinepelloux.eu-php.conf" /etc/apache2/sites-available/004-traider.antoinepelloux.eu.conf
echo "   ✓ Configuration copied"

# Step 4: Enable the site
echo "4. Enabling site..."
sudo a2ensite 004-traider.antoinepelloux.eu.conf
echo "   ✓ Site enabled"

# Step 5: Test Apache configuration
echo "5. Testing Apache configuration..."
if sudo apache2ctl configtest; then
    echo "   ✓ Apache config is valid"
else
    echo "   ✗ Apache config test failed!"
    exit 1
fi

# Step 6: Check .env file
echo "6. Checking environment configuration..."
if [ ! -f "$SCRIPT_DIR/.env" ]; then
    echo "   ! Creating .env from template..."
    cp "$SCRIPT_DIR/.env.example" "$SCRIPT_DIR/.env"
    echo "   ⚠ You MUST edit .env and configure your credentials!"
else
    echo "   ✓ .env file exists"
fi

# Step 7: Set proper permissions
echo "7. Setting file permissions..."
chmod 755 "$SCRIPT_DIR/api"
chmod 644 "$SCRIPT_DIR/api/"*.php
chmod 755 "$SCRIPT_DIR/cron"
chmod 755 "$SCRIPT_DIR/cron/"*.php
chmod 600 "$SCRIPT_DIR/.env"
echo "   ✓ Permissions set"

# Step 8: Initialize database
echo "8. Initializing database..."
php -r "
require_once '$SCRIPT_DIR/config/config.php';
require_once '$SCRIPT_DIR/config/database.php';
try {
    \$db = Database::getInstance();
    \$db->initializeTables();
    echo '   ✓ Database tables created\n';
} catch (Exception \$e) {
    echo '   ✗ Database initialization failed: ' . \$e->getMessage() . '\n';
    exit(1);
}
"

# Step 9: Set up cron jobs
echo "9. Setting up cron jobs..."
echo "   Adding cron entries..."

# Create cron entries
CRON_PREDICTION="0 18 * * * /usr/bin/php $SCRIPT_DIR/cron/daily-prediction.php >> /var/log/stock-predictor-prediction.log 2>&1"
CRON_SCORING="0 9 * * * /usr/bin/php $SCRIPT_DIR/cron/daily-scoring.php >> /var/log/stock-predictor-scoring.log 2>&1"

# Add to crontab
(crontab -l 2>/dev/null | grep -v "stock-predictor"; echo "$CRON_PREDICTION"; echo "$CRON_SCORING") | crontab -

echo "   ✓ Cron jobs scheduled"
echo "     - Daily Prediction: 18:00 CET"
echo "     - Daily Scoring: 09:00 CET"

# Step 10: Reload Apache
echo "10. Reloading Apache..."
sudo systemctl reload apache2
echo "   ✓ Apache reloaded"

echo ""
echo "========================================"
echo "Installation Complete!"
echo "========================================"
echo ""
echo "Your site should be available at:"
echo "  → http://traider.antoinepelloux.eu"
echo ""
echo "⚠ IMPORTANT NEXT STEPS:"
echo "1. Edit .env and configure:"
echo "   - ANTHROPIC_API_KEY"
echo "   - MySQL credentials (DB_USER, DB_PASSWORD, etc.)"
echo ""
echo "2. Make sure MySQL is set up:"
echo "   sudo mysql -u root -p < setup-database.sql"
echo ""
echo "3. Test the installation:"
echo "   curl http://traider.antoinepelloux.eu/api/health"
echo ""
echo "Cron Job Logs:"
echo "  tail -f /var/log/stock-predictor-prediction.log"
echo "  tail -f /var/log/stock-predictor-scoring.log"
echo ""
echo "Apache Logs:"
echo "  tail -f /var/log/apache2/traider-error.log"
echo ""
