// API Base URL
const API_BASE = '/api';

// Initialize app
document.addEventListener('DOMContentLoaded', () => {
    initializeApp();
    setupEventListeners();
});

/**
 * Initialize the application
 */
async function initializeApp() {
    try {
        await checkServerStatus();
        await loadAllData();
    } catch (error) {
        console.error('Error initializing app:', error);
        showError('Failed to initialize application');
    }
}

/**
 * Setup event listeners
 */
function setupEventListeners() {
    document.getElementById('refreshBtn').addEventListener('click', async () => {
        await loadAllData();
    });

    document.getElementById('runPredictionBtn').addEventListener('click', async () => {
        if (confirm('This will run a new prediction. It may take a few minutes. Continue?')) {
            await runNewPrediction();
        }
    });
}

/**
 * Check server status
 */
async function checkServerStatus() {
    try {
        const response = await fetch(`${API_BASE}/health`);
        const data = await response.json();

        const statusElement = document.getElementById('serverStatus');
        statusElement.textContent = data.status === 'ok' ? 'Online' : 'Offline';
        statusElement.className = `status-value status-badge ${data.status === 'ok' ? 'online' : 'offline'}`;

        updateLastUpdate();
    } catch (error) {
        document.getElementById('serverStatus').textContent = 'Offline';
        document.getElementById('serverStatus').className = 'status-value status-badge offline';
    }
}

/**
 * Load all data
 */
async function loadAllData() {
    await Promise.all([
        loadPredictions(),
        loadScoreHistory(),
        loadMarketData(),
        loadNews()
    ]);
    updateLastUpdate();
}

/**
 * Load latest predictions
 */
async function loadPredictions() {
    try {
        const response = await fetch(`${API_BASE}/predictions`);
        const data = await response.json();

        if (data.success) {
            displayPredictions(data);
        } else {
            document.getElementById('predictionsUp').innerHTML = '<div class="loading">No predictions available yet</div>';
            document.getElementById('predictionsDown').innerHTML = '<div class="loading">No predictions available yet</div>';
        }
    } catch (error) {
        console.error('Error loading predictions:', error);
        document.getElementById('predictionsUp').innerHTML = '<div class="loading">Error loading predictions</div>';
        document.getElementById('predictionsDown').innerHTML = '<div class="loading">Error loading predictions</div>';
    }
}

/**
 * Display predictions
 */
function displayPredictions(data) {
    const predictionDate = document.getElementById('predictionDate');
    predictionDate.textContent = data.prediction_date
        ? `Predictions made on: ${formatDate(data.prediction_date)}`
        : 'No predictions available';

    // Display UP predictions
    const predictionsUp = document.getElementById('predictionsUp');
    if (data.predictions_up && data.predictions_up.length > 0) {
        predictionsUp.innerHTML = data.predictions_up
            .map(pred => createPredictionHTML(pred, 'up'))
            .join('');
    } else {
        predictionsUp.innerHTML = '<div class="loading">No predictions available</div>';
    }

    // Display DOWN predictions
    const predictionsDown = document.getElementById('predictionsDown');
    if (data.predictions_down && data.predictions_down.length > 0) {
        predictionsDown.innerHTML = data.predictions_down
            .map(pred => createPredictionHTML(pred, 'down'))
            .join('');
    } else {
        predictionsDown.innerHTML = '<div class="loading">No predictions available</div>';
    }
}

/**
 * Create prediction HTML
 */
function createPredictionHTML(pred, type) {
    const hasScore = pred.correct !== null;
    const correctClass = hasScore ? (pred.correct ? 'correct' : 'incorrect') : '';

    return `
        <div class="prediction-item ${type} ${correctClass}">
            <div class="prediction-header">
                <span class="prediction-rank">#${pred.rank}</span>
                <span class="prediction-title">${pred.name}</span>
                <span class="prediction-symbol">${pred.symbol}</span>
            </div>
            <div class="prediction-reasoning">${pred.reasoning}</div>
            ${hasScore ? `
                <div class="prediction-result ${pred.correct ? 'correct' : 'incorrect'}">
                    ${pred.correct ? '✓ Correct' : '✗ Incorrect'}
                    (Actual: ${pred.actual_change_percent > 0 ? '+' : ''}${pred.actual_change_percent?.toFixed(2)}%)
                </div>
            ` : ''}
        </div>
    `;
}

/**
 * Load score history
 */
async function loadScoreHistory() {
    try {
        const response = await fetch(`${API_BASE}/scores?limit=30`);
        const data = await response.json();

        if (data.success && data.data.length > 0) {
            displayScoreHistory(data.data);

            // Display current score (most recent)
            const latestScore = data.data[0];
            document.getElementById('overallAccuracy').textContent =
                `${latestScore.accuracy_percent.toFixed(1)}%`;
            document.getElementById('correctCount').textContent =
                `${latestScore.correct_predictions}`;
            document.getElementById('totalCount').textContent =
                `${latestScore.total_predictions}`;
        } else {
            document.getElementById('overallAccuracy').textContent = '--';
            document.getElementById('correctCount').textContent = '--';
            document.getElementById('totalCount').textContent = '--';
            document.getElementById('scoreHistory').innerHTML =
                '<div class="loading">No historical data available yet</div>';
        }
    } catch (error) {
        console.error('Error loading scores:', error);
    }
}

/**
 * Display score history
 */
function displayScoreHistory(scores) {
    const historyContainer = document.getElementById('scoreHistory');

    historyContainer.innerHTML = scores
        .map(score => {
            const accuracyClass = score.accuracy_percent >= 70 ? 'high' :
                                 score.accuracy_percent >= 50 ? 'medium' : 'low';

            return `
                <div class="history-item ${accuracyClass}">
                    <div>
                        <div class="history-date">${formatDate(score.date)}</div>
                        <div class="history-details">
                            ${score.correct_predictions} / ${score.total_predictions} correct
                        </div>
                    </div>
                    <div class="history-accuracy">${score.accuracy_percent.toFixed(1)}%</div>
                </div>
            `;
        })
        .join('');
}

/**
 * Load market data
 */
async function loadMarketData() {
    try {
        const response = await fetch(`${API_BASE}/market-data`);
        const data = await response.json();

        if (data.success) {
            displayMarketData(data.data);
        }
    } catch (error) {
        console.error('Error loading market data:', error);
        document.getElementById('marketData').innerHTML =
            '<div class="loading">Error loading market data</div>';
    }
}

/**
 * Display market data
 */
function displayMarketData(markets) {
    const marketContainer = document.getElementById('marketData');

    if (markets.length === 0) {
        marketContainer.innerHTML = '<div class="loading">No market data available</div>';
        return;
    }

    marketContainer.innerHTML = markets
        .map(market => {
            const changeClass = market.changePercent >= 0 ? 'positive' : 'negative';
            const changeSymbol = market.changePercent >= 0 ? '+' : '';

            return `
                <div class="market-item">
                    <div class="market-header">
                        <div>
                            <div class="market-name">${market.name}</div>
                            <div class="market-country">${market.country}</div>
                        </div>
                        <div class="market-symbol">${market.symbol}</div>
                    </div>
                    <div class="market-price">${market.close?.toFixed(2) || 'N/A'}</div>
                    <div class="market-change ${changeClass}">
                        ${changeSymbol}${market.changePercent?.toFixed(2) || 'N/A'}%
                    </div>
                </div>
            `;
        })
        .join('');
}

/**
 * Load news
 */
async function loadNews() {
    try {
        const response = await fetch(`${API_BASE}/news`);
        const data = await response.json();

        if (data.success) {
            displayNews(data.data);
        }
    } catch (error) {
        console.error('Error loading news:', error);
        document.getElementById('newsFeed').innerHTML =
            '<div class="loading">Error loading news</div>';
    }
}

/**
 * Display news
 */
function displayNews(articles) {
    const newsContainer = document.getElementById('newsFeed');

    if (articles.length === 0) {
        newsContainer.innerHTML = '<div class="loading">No news available</div>';
        return;
    }

    newsContainer.innerHTML = articles
        .slice(0, 20)
        .map(article => `
            <div class="news-item">
                <div class="news-title">
                    <a href="${article.link}" target="_blank" rel="noopener noreferrer">
                        ${article.title}
                    </a>
                </div>
                <div class="news-meta">
                    ${article.source} • ${formatDate(article.pubDate)}
                </div>
            </div>
        `)
        .join('');
}

/**
 * Run new prediction
 */
async function runNewPrediction() {
    const btn = document.getElementById('runPredictionBtn');
    btn.disabled = true;
    btn.textContent = 'Running...';

    try {
        const response = await fetch(`${API_BASE}/run-prediction`, {
            method: 'POST'
        });
        const data = await response.json();

        if (data.success) {
            alert('Prediction completed successfully!');
            await loadAllData();
        } else {
            alert('Prediction failed: ' + data.error);
        }
    } catch (error) {
        console.error('Error running prediction:', error);
        alert('Failed to run prediction: ' + error.message);
    } finally {
        btn.disabled = false;
        btn.textContent = 'Run New Prediction';
    }
}

/**
 * Format date string
 */
function formatDate(dateString) {
    if (!dateString) return 'N/A';

    try {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    } catch (error) {
        return dateString;
    }
}

/**
 * Update last update timestamp
 */
function updateLastUpdate() {
    const now = new Date();
    document.getElementById('lastUpdate').textContent = now.toLocaleTimeString();
}

/**
 * Show error message
 */
function showError(message) {
    console.error(message);
    // You can implement a toast notification here
}
