<?php
/**
 * Claude AI Service
 * Generates stock market predictions using Anthropic's Claude AI
 */

class ClaudeAIService {

    private $apiKey;
    private $apiUrl = 'https://api.anthropic.com/v1/messages';

    public function __construct() {
        $this->apiKey = ANTHROPIC_API_KEY;
    }

    /**
     * Generate predictions using Claude AI
     */
    public function generatePredictions($marketData, $newsArticles) {
        try {
            $marketSummary = $this->formatMarketData($marketData);
            $newsSummary = $this->formatNewsArticles($newsArticles);

            $prompt = "You are an expert financial analyst specializing in European stock markets. Based on the following market data and recent news, predict which European stock indices will perform best and worst tomorrow.

{$marketSummary}

{$newsSummary}

Please analyze this information and provide:
1. Top 5 indices that are most likely to go UP tomorrow
2. Top 5 indices that are most likely to go DOWN tomorrow

For each prediction, provide:
- The index symbol and name
- A brief reasoning (2-3 sentences) explaining why you think it will move in that direction
- Consider technical indicators, recent news, economic trends, and market sentiment

Format your response as JSON:
{
  \"predictions_up\": [
    {
      \"symbol\": \"^FTSE\",
      \"name\": \"FTSE 100\",
      \"reasoning\": \"...\"
    }
  ],
  \"predictions_down\": [
    {
      \"symbol\": \"^GDAXI\",
      \"name\": \"DAX\",
      \"reasoning\": \"...\"
    }
  ]
}";

            $response = $this->callClaudeAPI($prompt);

            // Extract JSON from response
            if (preg_match('/\{[\s\S]*\}/', $response, $matches)) {
                $predictions = json_decode($matches[0], true);

                if ($predictions) {
                    // Add rank and type to predictions
                    foreach ($predictions['predictions_up'] as $idx => &$pred) {
                        $pred['rank'] = $idx + 1;
                        $pred['type'] = 'UP';
                    }

                    foreach ($predictions['predictions_down'] as $idx => &$pred) {
                        $pred['rank'] = $idx + 1;
                        $pred['type'] = 'DOWN';
                    }

                    return $predictions;
                }
            }

            throw new Exception("Could not parse JSON from Claude response");

        } catch (Exception $e) {
            error_log("Error generating predictions with Claude: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Call Claude API
     */
    private function callClaudeAPI($prompt) {
        $data = [
            'model' => 'claude-3-5-sonnet-20241022',
            'max_tokens' => 2000,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ]
        ];

        $ch = curl_init($this->apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'x-api-key: ' . $this->apiKey,
            'anthropic-version: 2023-06-01'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new Exception("Claude API request failed: HTTP $httpCode");
        }

        $result = json_decode($response, true);

        if (!isset($result['content'][0]['text'])) {
            throw new Exception("Invalid response from Claude API");
        }

        return $result['content'][0]['text'];
    }

    /**
     * Format market data for Claude
     */
    private function formatMarketData($marketData) {
        $formatted = "Current Market Data (European Indices):\n\n";

        foreach ($marketData as $data) {
            $changeSymbol = $data['changePercent'] >= 0 ? '+' : '';
            $formatted .= $data['name'] . " (" . $data['symbol'] . ") - " . $data['country'] . "\n";
            $formatted .= "  Current: " . number_format($data['close'], 2) . "\n";
            $formatted .= "  Change: " . $changeSymbol . number_format($data['changePercent'], 2) . "%\n";
            $formatted .= "  Volume: " . number_format($data['volume']) . "\n";
            $formatted .= "\n";
        }

        return $formatted;
    }

    /**
     * Format news articles for Claude
     */
    private function formatNewsArticles($newsArticles) {
        if (empty($newsArticles)) {
            return "No recent news articles available.";
        }

        $formatted = "Recent Financial News:\n\n";

        foreach (array_slice($newsArticles, 0, 15) as $index => $article) {
            $formatted .= ($index + 1) . ". " . $article['title'] . "\n";
            $formatted .= "   Source: " . $article['source'] . " | Published: " . $article['pubDate'] . "\n";

            if (!empty($article['contentSnippet'])) {
                $snippet = substr($article['contentSnippet'], 0, 200);
                $formatted .= "   " . $snippet . "...\n";
            }

            $formatted .= "\n";
        }

        return $formatted;
    }
}
