<?php
/**
 * Google News Service
 * Fetches financial news from Google News RSS feeds
 */

class GoogleNewsService {

    /**
     * Fetch news articles from Google News RSS feed
     */
    public function fetchFinanceNews($query = 'European stock market', $maxArticles = 20) {
        try {
            $searchQuery = urlencode($query);
            $rssUrl = "https://news.google.com/rss/search?q={$searchQuery}&hl=en-US&gl=US&ceid=US:en";

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $rssUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200 || !$response) {
                error_log("Failed to fetch news: HTTP $httpCode");
                return [];
            }

            // Parse XML
            $xml = simplexml_load_string($response);
            if (!$xml) {
                error_log("Failed to parse RSS feed");
                return [];
            }

            $articles = [];
            $count = 0;

            foreach ($xml->channel->item as $item) {
                if ($count >= $maxArticles) {
                    break;
                }

                $articles[] = [
                    'title' => (string) $item->title,
                    'link' => (string) $item->link,
                    'pubDate' => (string) $item->pubDate,
                    'source' => isset($item->source) ? (string) $item->source : 'Unknown',
                    'contentSnippet' => isset($item->description) ? strip_tags((string) $item->description) : ''
                ];

                $count++;
            }

            return $articles;

        } catch (Exception $e) {
            error_log("Error fetching Google News: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Fetch news for multiple European stock markets
     */
    public function fetchEuropeanMarketNews() {
        $queries = [
            'European stock market',
            'FTSE 100',
            'DAX Germany',
            'CAC 40 France',
            'European economy',
            'European Central Bank'
        ];

        $allArticles = [];

        foreach ($queries as $query) {
            $articles = $this->fetchFinanceNews($query, 5);
            $allArticles = array_merge($allArticles, $articles);

            // Small delay to avoid rate limiting
            sleep(1);
        }

        // Remove duplicates based on title
        $uniqueArticles = [];
        $titles = [];

        foreach ($allArticles as $article) {
            if (!in_array($article['title'], $titles)) {
                $titles[] = $article['title'];
                $uniqueArticles[] = $article;
            }
        }

        return $uniqueArticles;
    }

    /**
     * Format news articles for Claude AI
     */
    public function formatNewsForClaude($articles) {
        if (empty($articles)) {
            return "No recent news articles available.";
        }

        $formatted = "Recent Financial News:\n\n";

        foreach (array_slice($articles, 0, 15) as $index => $article) {
            $formatted .= ($index + 1) . ". " . $article['title'] . "\n";
            $formatted .= "   Source: " . $article['source'] . "\n";
            $formatted .= "   Published: " . $article['pubDate'] . "\n";

            if (!empty($article['contentSnippet'])) {
                $snippet = substr($article['contentSnippet'], 0, 200);
                $formatted .= "   " . $snippet . "...\n";
            }

            $formatted .= "\n";
        }

        return $formatted;
    }
}
