<?php
/**
 * Scoring Service
 * Handles saving market data, predictions, and scoring
 */

class ScoringService {

    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Save market data to database
     */
    public function saveMarketData($date, $marketData) {
        $sql = "INSERT INTO market_data
                (date, symbol, name, open_price, close_price, high_price, low_price, volume, change_percent)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    open_price = VALUES(open_price),
                    close_price = VALUES(close_price),
                    high_price = VALUES(high_price),
                    low_price = VALUES(low_price),
                    volume = VALUES(volume),
                    change_percent = VALUES(change_percent)";

        $saved = 0;
        foreach ($marketData as $data) {
            try {
                $this->db->execute($sql, [
                    $date,
                    $data['symbol'],
                    $data['name'],
                    $data['open'],
                    $data['close'],
                    $data['high'],
                    $data['low'],
                    $data['volume'],
                    $data['changePercent']
                ]);
                $saved++;
            } catch (Exception $e) {
                error_log("Error saving market data: " . $e->getMessage());
            }
        }

        return $saved;
    }

    /**
     * Save news articles to database
     */
    public function saveNewsArticles($date, $articles) {
        if (empty($articles)) {
            return 0;
        }

        $sql = "INSERT INTO news_articles (date, title, link, source, pub_date)
                VALUES (?, ?, ?, ?, ?)";

        $saved = 0;
        foreach ($articles as $article) {
            try {
                $this->db->execute($sql, [
                    $date,
                    $article['title'],
                    $article['link'],
                    $article['source'],
                    $article['pubDate']
                ]);
                $saved++;
            } catch (Exception $e) {
                error_log("Error saving news article: " . $e->getMessage());
            }
        }

        return $saved;
    }

    /**
     * Save predictions to database
     */
    public function savePredictions($predictionDate, $targetDate, $predictions) {
        $sql = "INSERT INTO predictions
                (prediction_date, target_date, prediction_type, symbol, name, rank, reasoning)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    rank = VALUES(rank),
                    reasoning = VALUES(reasoning)";

        $allPredictions = array_merge(
            $predictions['predictions_up'],
            $predictions['predictions_down']
        );

        $saved = 0;
        foreach ($allPredictions as $pred) {
            try {
                $this->db->execute($sql, [
                    $predictionDate,
                    $targetDate,
                    $pred['type'],
                    $pred['symbol'],
                    $pred['name'],
                    $pred['rank'],
                    $pred['reasoning']
                ]);
                $saved++;
            } catch (Exception $e) {
                error_log("Error saving prediction: " . $e->getMessage());
            }
        }

        return $saved;
    }

    /**
     * Score predictions against actual market performance
     */
    public function scorePredictions($predictionDate, $actualDate) {
        // Get predictions for the given date
        $predictions = $this->db->fetchAll(
            "SELECT * FROM predictions WHERE prediction_date = ?",
            [$predictionDate]
        );

        if (empty($predictions)) {
            return [
                'date' => $actualDate,
                'total_predictions' => 0,
                'correct_predictions' => 0,
                'accuracy_percent' => 0
            ];
        }

        $correctCount = 0;
        $totalCount = count($predictions);

        foreach ($predictions as $prediction) {
            $actualData = $this->getMarketData($actualDate, $prediction['symbol']);

            if ($actualData && $actualData['change_percent'] !== null) {
                $predictedUp = $prediction['prediction_type'] === 'UP';
                $actualUp = $actualData['change_percent'] > 0;
                $correct = $predictedUp === $actualUp;

                if ($correct) {
                    $correctCount++;
                }

                // Save individual prediction score
                $this->savePredictionScore(
                    $prediction['id'],
                    $actualData['change_percent'],
                    $correct,
                    $actualDate
                );
            }
        }

        $accuracy = $totalCount > 0 ? ($correctCount / $totalCount) * 100 : 0;

        // Save daily score
        $this->saveDailyScore($actualDate, $totalCount, $correctCount, $accuracy);

        return [
            'date' => $actualDate,
            'total_predictions' => $totalCount,
            'correct_predictions' => $correctCount,
            'accuracy_percent' => $accuracy
        ];
    }

    /**
     * Get market data for a specific date and symbol
     */
    public function getMarketData($date, $symbol) {
        return $this->db->fetchOne(
            "SELECT * FROM market_data WHERE date = ? AND symbol = ?",
            [$date, $symbol]
        );
    }

    /**
     * Save individual prediction score
     */
    private function savePredictionScore($predictionId, $actualChange, $correct, $scoreDate) {
        $sql = "INSERT INTO prediction_scores
                (prediction_id, actual_change_percent, correct, score_date)
                VALUES (?, ?, ?, ?)";

        $this->db->execute($sql, [
            $predictionId,
            $actualChange,
            $correct ? 1 : 0,
            $scoreDate
        ]);
    }

    /**
     * Save daily overall score
     */
    private function saveDailyScore($date, $total, $correct, $accuracy) {
        $sql = "INSERT INTO daily_scores
                (date, total_predictions, correct_predictions, accuracy_percent)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    total_predictions = VALUES(total_predictions),
                    correct_predictions = VALUES(correct_predictions),
                    accuracy_percent = VALUES(accuracy_percent)";

        $this->db->execute($sql, [$date, $total, $correct, $accuracy]);
    }

    /**
     * Get historical scores
     */
    public function getHistoricalScores($limit = 30) {
        return $this->db->fetchAll(
            "SELECT * FROM daily_scores ORDER BY date DESC LIMIT ?",
            [(int)$limit]
        );
    }

    /**
     * Get latest predictions with scores
     */
    public function getLatestPredictionsWithScores() {
        $sql = "SELECT
                    p.*,
                    ps.actual_change_percent,
                    ps.correct,
                    ps.score_date
                FROM predictions p
                LEFT JOIN prediction_scores ps ON p.id = ps.prediction_id
                WHERE p.prediction_date = (SELECT MAX(prediction_date) FROM predictions)
                ORDER BY p.prediction_type, p.rank";

        return $this->db->fetchAll($sql);
    }
}
