-- Stock Market Predictor - MySQL Database Setup Script
-- Run this script as MySQL root user to create the database and user

-- Create database
CREATE DATABASE IF NOT EXISTS stock_predictor
  CHARACTER SET utf8mb4
  COLLATE utf8mb4_unicode_ci;

-- Create user (change password to something secure)
CREATE USER IF NOT EXISTS 'stock_predictor_user'@'localhost'
  IDENTIFIED BY 'your_secure_password_here';

-- Grant privileges
GRANT ALL PRIVILEGES ON stock_predictor.* TO 'stock_predictor_user'@'localhost';

-- Apply privileges
FLUSH PRIVILEGES;

-- Use the database
USE stock_predictor;

-- Tables will be created automatically by the Node.js application
-- But you can run these manually if needed:

-- Table for storing daily market data
CREATE TABLE IF NOT EXISTS market_data (
  id INT AUTO_INCREMENT PRIMARY KEY,
  date DATE NOT NULL,
  symbol VARCHAR(50) NOT NULL,
  name VARCHAR(100) NOT NULL,
  open_price DECIMAL(10, 2),
  close_price DECIMAL(10, 2),
  high_price DECIMAL(10, 2),
  low_price DECIMAL(10, 2),
  volume BIGINT,
  change_percent DECIMAL(10, 4),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  UNIQUE KEY unique_date_symbol (date, symbol),
  INDEX idx_date (date),
  INDEX idx_symbol (symbol)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for storing news articles
CREATE TABLE IF NOT EXISTS news_articles (
  id INT AUTO_INCREMENT PRIMARY KEY,
  date DATE NOT NULL,
  title VARCHAR(500) NOT NULL,
  link VARCHAR(1000),
  source VARCHAR(200),
  pub_date VARCHAR(100),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  INDEX idx_date (date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for storing predictions
CREATE TABLE IF NOT EXISTS predictions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  prediction_date DATE NOT NULL,
  target_date DATE NOT NULL,
  prediction_type VARCHAR(10) NOT NULL,
  symbol VARCHAR(50) NOT NULL,
  name VARCHAR(100) NOT NULL,
  rank INT NOT NULL,
  reasoning TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  UNIQUE KEY unique_prediction (prediction_date, prediction_type, symbol),
  INDEX idx_prediction_date (prediction_date),
  INDEX idx_target_date (target_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for storing prediction scores
CREATE TABLE IF NOT EXISTS prediction_scores (
  id INT AUTO_INCREMENT PRIMARY KEY,
  prediction_id INT NOT NULL,
  actual_change_percent DECIMAL(10, 4),
  correct BOOLEAN,
  score_date DATE NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (prediction_id) REFERENCES predictions(id) ON DELETE CASCADE,
  INDEX idx_prediction_id (prediction_id),
  INDEX idx_score_date (score_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table for storing daily overall scores
CREATE TABLE IF NOT EXISTS daily_scores (
  id INT AUTO_INCREMENT PRIMARY KEY,
  date DATE NOT NULL UNIQUE,
  total_predictions INT,
  correct_predictions INT,
  accuracy_percent DECIMAL(5, 2),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  INDEX idx_date (date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Show tables
SHOW TABLES;

SELECT 'Database setup completed successfully!' AS Status;
