#!/bin/bash

# Complete Production Setup for Stock Market Predictor
# Run this script to set up Apache and systemd service

set -e  # Exit on error

echo "=========================================="
echo "Stock Market Predictor - Production Setup"
echo "=========================================="
echo ""

# Get the script directory
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

echo "Working directory: $SCRIPT_DIR"
echo ""

# Step 1: Copy Apache configuration
echo "1. Installing Apache configuration..."
if [ -f "$SCRIPT_DIR/apache-config/004-traider.antoinepelloux.eu.conf" ]; then
    sudo cp "$SCRIPT_DIR/apache-config/004-traider.antoinepelloux.eu.conf" /etc/apache2/sites-available/
    echo "   ✓ Apache config copied"
else
    echo "   ✗ Apache config file not found!"
    exit 1
fi

# Step 2: Enable required Apache modules
echo "2. Enabling Apache modules..."
sudo a2enmod proxy proxy_http rewrite headers deflate expires 2>/dev/null || true
echo "   ✓ Modules enabled"

# Step 3: Enable site
echo "3. Enabling site..."
sudo a2ensite 004-traider.antoinepelloux.eu.conf
echo "   ✓ Site enabled"

# Step 4: Test Apache configuration
echo "4. Testing Apache configuration..."
if sudo apache2ctl configtest; then
    echo "   ✓ Apache config is valid"
else
    echo "   ✗ Apache config test failed!"
    exit 1
fi

# Step 5: Install systemd service
echo "5. Installing systemd service..."
if [ -f "$SCRIPT_DIR/apache-config/stock-predictor.service" ]; then
    sudo cp "$SCRIPT_DIR/apache-config/stock-predictor.service" /etc/systemd/system/
    sudo systemctl daemon-reload
    echo "   ✓ Systemd service installed"
else
    echo "   ✗ Service file not found!"
    exit 1
fi

# Step 6: Check if .env exists
echo "6. Checking environment configuration..."
if [ -f "$SCRIPT_DIR/.env" ]; then
    echo "   ✓ .env file exists"

    # Check if API key is configured
    if grep -q "your_anthropic_api_key_here" "$SCRIPT_DIR/.env"; then
        echo "   ⚠ WARNING: You need to configure your API key in .env"
    fi

    # Check if MySQL password is configured
    if grep -q "your_secure_password_here" "$SCRIPT_DIR/.env"; then
        echo "   ⚠ WARNING: You need to configure MySQL credentials in .env"
    fi
else
    echo "   ⚠ .env file not found! Creating from template..."
    cp "$SCRIPT_DIR/.env.example" "$SCRIPT_DIR/.env"
    echo "   ! You MUST edit .env and configure your credentials!"
fi

# Step 7: Check Node.js dependencies
echo "7. Checking Node.js dependencies..."
if [ -d "$SCRIPT_DIR/node_modules" ]; then
    echo "   ✓ Dependencies installed"
else
    echo "   ! Dependencies not installed"
    echo "   Installing now..."
    cd "$SCRIPT_DIR"
    npm install
    echo "   ✓ Dependencies installed"
fi

# Step 8: Enable and start Node.js service
echo "8. Starting Node.js service..."
sudo systemctl enable stock-predictor
sudo systemctl restart stock-predictor
sleep 2

if sudo systemctl is-active --quiet stock-predictor; then
    echo "   ✓ Service is running"
else
    echo "   ✗ Service failed to start!"
    echo "   Checking logs..."
    sudo journalctl -u stock-predictor -n 20 --no-pager
fi

# Step 9: Reload Apache
echo "9. Reloading Apache..."
sudo systemctl reload apache2
echo "   ✓ Apache reloaded"

echo ""
echo "=========================================="
echo "Setup Complete!"
echo "=========================================="
echo ""
echo "Your site should be available at:"
echo "  → http://traider.antoinepelloux.eu"
echo ""
echo "Service Management:"
echo "  Start:   sudo systemctl start stock-predictor"
echo "  Stop:    sudo systemctl stop stock-predictor"
echo "  Restart: sudo systemctl restart stock-predictor"
echo "  Status:  sudo systemctl status stock-predictor"
echo "  Logs:    sudo journalctl -u stock-predictor -f"
echo ""
echo "Apache Logs:"
echo "  Error:   sudo tail -f /var/log/apache2/traider-error.log"
echo "  Access:  sudo tail -f /var/log/apache2/traider-access.log"
echo ""

# Check service status
echo "Current Service Status:"
sudo systemctl status stock-predictor --no-pager -l | head -20
echo ""

# Final checks
echo "Final Checks:"
echo -n "  Apache: "
if sudo systemctl is-active --quiet apache2; then
    echo "✓ Running"
else
    echo "✗ Not running"
fi

echo -n "  Node.js: "
if sudo systemctl is-active --quiet stock-predictor; then
    echo "✓ Running"
else
    echo "✗ Not running"
fi

echo -n "  Port 3000: "
if netstat -tuln | grep -q ":3000 "; then
    echo "✓ Listening"
else
    echo "✗ Not listening"
fi

echo ""
echo "⚠ IMPORTANT: Make sure to edit .env with your credentials!"
echo ""
