const mysql = require('mysql2/promise');

// Database connection configuration
const dbConfig = {
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || 'stock_predictor',
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};

// Create connection pool
const pool = mysql.createPool(dbConfig);

/**
 * Initialize database tables
 */
async function initializeDatabase() {
  try {
    const connection = await pool.getConnection();
    console.log('Connected to MySQL database');

    // Table for storing daily market data
    await connection.query(`
      CREATE TABLE IF NOT EXISTS market_data (
        id INT AUTO_INCREMENT PRIMARY KEY,
        date DATE NOT NULL,
        symbol VARCHAR(50) NOT NULL,
        name VARCHAR(100) NOT NULL,
        open_price DECIMAL(10, 2),
        close_price DECIMAL(10, 2),
        high_price DECIMAL(10, 2),
        low_price DECIMAL(10, 2),
        volume BIGINT,
        change_percent DECIMAL(10, 4),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_date_symbol (date, symbol),
        INDEX idx_date (date),
        INDEX idx_symbol (symbol)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);

    // Table for storing news articles
    await connection.query(`
      CREATE TABLE IF NOT EXISTS news_articles (
        id INT AUTO_INCREMENT PRIMARY KEY,
        date DATE NOT NULL,
        title VARCHAR(500) NOT NULL,
        link VARCHAR(1000),
        source VARCHAR(200),
        pub_date VARCHAR(100),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_date (date)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);

    // Table for storing predictions
    await connection.query(`
      CREATE TABLE IF NOT EXISTS predictions (
        id INT AUTO_INCREMENT PRIMARY KEY,
        prediction_date DATE NOT NULL,
        target_date DATE NOT NULL,
        prediction_type VARCHAR(10) NOT NULL,
        symbol VARCHAR(50) NOT NULL,
        name VARCHAR(100) NOT NULL,
        rank INT NOT NULL,
        reasoning TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY unique_prediction (prediction_date, prediction_type, symbol),
        INDEX idx_prediction_date (prediction_date),
        INDEX idx_target_date (target_date)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);

    // Table for storing prediction scores
    await connection.query(`
      CREATE TABLE IF NOT EXISTS prediction_scores (
        id INT AUTO_INCREMENT PRIMARY KEY,
        prediction_id INT NOT NULL,
        actual_change_percent DECIMAL(10, 4),
        correct BOOLEAN,
        score_date DATE NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (prediction_id) REFERENCES predictions(id) ON DELETE CASCADE,
        INDEX idx_prediction_id (prediction_id),
        INDEX idx_score_date (score_date)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);

    // Table for storing daily overall scores
    await connection.query(`
      CREATE TABLE IF NOT EXISTS daily_scores (
        id INT AUTO_INCREMENT PRIMARY KEY,
        date DATE NOT NULL UNIQUE,
        total_predictions INT,
        correct_predictions INT,
        accuracy_percent DECIMAL(5, 2),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_date (date)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    `);

    console.log('Database tables initialized successfully');
    connection.release();
  } catch (error) {
    console.error('Error initializing database:', error.message);
    throw error;
  }
}

// Initialize database on startup
initializeDatabase().catch(err => {
  console.error('Failed to initialize database:', err);
});

/**
 * Execute a query and return results
 */
async function query(sql, params = []) {
  try {
    const [results] = await pool.execute(sql, params);
    return results;
  } catch (error) {
    console.error('Database query error:', error.message);
    throw error;
  }
}

/**
 * Close database connection pool
 */
async function close() {
  try {
    await pool.end();
    console.log('Database connection pool closed');
  } catch (error) {
    console.error('Error closing database pool:', error.message);
  }
}

module.exports = {
  pool,
  query,
  close,
  initializeDatabase
};
