const express = require('express');
const router = express.Router();
const { runDailyPrediction, runDailyScoring } = require('../services/scheduler');
const { fetchAllEuropeanIndices } = require('../services/yahooFinance');
const { fetchEuropeanMarketNews } = require('../services/googleNews');
const {
  getHistoricalScores,
  getLatestPredictionsWithScores
} = require('../services/scoring');

/**
 * GET /api/health
 * Health check endpoint
 */
router.get('/health', (req, res) => {
  res.json({
    status: 'ok',
    timestamp: new Date().toISOString()
  });
});

/**
 * GET /api/market-data
 * Get current European market data
 */
router.get('/market-data', async (req, res) => {
  try {
    const marketData = await fetchAllEuropeanIndices();
    res.json({
      success: true,
      count: marketData.length,
      data: marketData,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/news
 * Get recent financial news
 */
router.get('/news', async (req, res) => {
  try {
    const news = await fetchEuropeanMarketNews();
    res.json({
      success: true,
      count: news.length,
      data: news,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/predictions/latest
 * Get the latest predictions with their scores
 */
router.get('/predictions/latest', async (req, res) => {
  try {
    const predictions = await getLatestPredictionsWithScores();

    const predictionsUp = predictions.filter(p => p.prediction_type === 'UP');
    const predictionsDown = predictions.filter(p => p.prediction_type === 'DOWN');

    res.json({
      success: true,
      prediction_date: predictions[0]?.prediction_date || null,
      predictions_up: predictionsUp,
      predictions_down: predictionsDown,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * GET /api/scores/history
 * Get historical accuracy scores
 */
router.get('/scores/history', async (req, res) => {
  try {
    const limit = parseInt(req.query.limit) || 30;
    const scores = await getHistoricalScores(limit);

    res.json({
      success: true,
      count: scores.length,
      data: scores,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * POST /api/predict/run
 * Manually trigger a prediction run
 */
router.post('/predict/run', async (req, res) => {
  try {
    console.log('Manual prediction run triggered via API');
    const result = await runDailyPrediction();
    res.json(result);
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

/**
 * POST /api/score/run
 * Manually trigger a scoring run
 */
router.post('/score/run', async (req, res) => {
  try {
    console.log('Manual scoring run triggered via API');
    const result = await runDailyScoring();
    res.json(result);
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
