const Anthropic = require('@anthropic-ai/sdk');

const client = new Anthropic({
  apiKey: process.env.ANTHROPIC_API_KEY
});

/**
 * Generate predictions using Claude AI
 */
async function generatePredictions(marketData, newsArticles) {
  try {
    // Format market data for Claude
    const marketSummary = formatMarketData(marketData);

    // Format news for Claude
    const newsSummary = formatNewsArticles(newsArticles);

    const prompt = `You are an expert financial analyst specializing in European stock markets. Based on the following market data and recent news, predict which European stock indices will perform best and worst tomorrow.

${marketSummary}

${newsSummary}

Please analyze this information and provide:
1. Top 5 indices that are most likely to go UP tomorrow
2. Top 5 indices that are most likely to go DOWN tomorrow

For each prediction, provide:
- The index symbol and name
- A brief reasoning (2-3 sentences) explaining why you think it will move in that direction
- Consider technical indicators, recent news, economic trends, and market sentiment

Format your response as JSON:
{
  "predictions_up": [
    {
      "symbol": "^FTSE",
      "name": "FTSE 100",
      "reasoning": "..."
    }
  ],
  "predictions_down": [
    {
      "symbol": "^GDAXI",
      "name": "DAX",
      "reasoning": "..."
    }
  ]
}`;

    const message = await client.messages.create({
      model: 'claude-3-5-sonnet-20241022',
      max_tokens: 2000,
      messages: [{
        role: 'user',
        content: prompt
      }]
    });

    // Extract JSON from response
    const responseText = message.content[0].text;
    const jsonMatch = responseText.match(/\{[\s\S]*\}/);

    if (!jsonMatch) {
      throw new Error('Could not parse JSON from Claude response');
    }

    const predictions = JSON.parse(jsonMatch[0]);

    // Add rank to each prediction
    predictions.predictions_up = predictions.predictions_up.map((pred, idx) => ({
      ...pred,
      rank: idx + 1,
      type: 'UP'
    }));

    predictions.predictions_down = predictions.predictions_down.map((pred, idx) => ({
      ...pred,
      rank: idx + 1,
      type: 'DOWN'
    }));

    return predictions;
  } catch (error) {
    console.error('Error generating predictions with Claude:', error.message);
    throw error;
  }
}

/**
 * Format market data for Claude
 */
function formatMarketData(marketData) {
  let formatted = "Current Market Data (European Indices):\n\n";

  marketData.forEach(data => {
    const changeSymbol = data.changePercent >= 0 ? '+' : '';
    formatted += `${data.name} (${data.symbol}) - ${data.country}\n`;
    formatted += `  Current: ${data.close?.toFixed(2) || 'N/A'}\n`;
    formatted += `  Change: ${changeSymbol}${data.changePercent?.toFixed(2) || 'N/A'}%\n`;
    formatted += `  Volume: ${data.volume?.toLocaleString() || 'N/A'}\n`;
    formatted += '\n';
  });

  return formatted;
}

/**
 * Format news articles for Claude
 */
function formatNewsArticles(newsArticles) {
  if (newsArticles.length === 0) {
    return "No recent news articles available.";
  }

  let formatted = "Recent Financial News:\n\n";
  newsArticles.slice(0, 15).forEach((article, index) => {
    formatted += `${index + 1}. ${article.title}\n`;
    formatted += `   Source: ${article.source} | Published: ${article.pubDate}\n`;
    if (article.contentSnippet) {
      formatted += `   ${article.contentSnippet.substring(0, 200)}...\n`;
    }
    formatted += '\n';
  });

  return formatted;
}

/**
 * Generate a daily summary report using Claude
 */
async function generateDailySummary(predictions, scores) {
  try {
    const prompt = `Generate a brief executive summary of today's stock market predictions and their accuracy.

Predictions Made:
- Top 5 predicted to go UP: ${predictions.predictions_up.map(p => p.name).join(', ')}
- Top 5 predicted to go DOWN: ${predictions.predictions_down.map(p => p.name).join(', ')}

Results:
- Total Predictions: ${scores.total_predictions}
- Correct Predictions: ${scores.correct_predictions}
- Accuracy: ${scores.accuracy_percent.toFixed(1)}%

Please provide a 2-3 paragraph summary analyzing the performance and any insights.`;

    const message = await client.messages.create({
      model: 'claude-3-5-sonnet-20241022',
      max_tokens: 500,
      messages: [{
        role: 'user',
        content: prompt
      }]
    });

    return message.content[0].text;
  } catch (error) {
    console.error('Error generating summary:', error.message);
    return 'Summary generation failed.';
  }
}

module.exports = {
  generatePredictions,
  generateDailySummary
};
