const { query } = require('../config/database');

/**
 * Save market data to database
 */
async function saveMarketData(date, marketData) {
  try {
    const sql = `
      INSERT INTO market_data
      (date, symbol, name, open_price, close_price, high_price, low_price, volume, change_percent)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        open_price = VALUES(open_price),
        close_price = VALUES(close_price),
        high_price = VALUES(high_price),
        low_price = VALUES(low_price),
        volume = VALUES(volume),
        change_percent = VALUES(change_percent)
    `;

    for (const data of marketData) {
      await query(sql, [
        date,
        data.symbol,
        data.name,
        data.open,
        data.close,
        data.high,
        data.low,
        data.volume,
        data.changePercent
      ]);
    }

    console.log(`Saved ${marketData.length} market data records`);
  } catch (error) {
    console.error('Error saving market data:', error.message);
    throw error;
  }
}

/**
 * Save news articles to database
 */
async function saveNewsArticles(date, articles) {
  try {
    if (articles.length === 0) return;

    const sql = `
      INSERT INTO news_articles (date, title, link, source, pub_date)
      VALUES (?, ?, ?, ?, ?)
    `;

    for (const article of articles) {
      await query(sql, [
        date,
        article.title,
        article.link,
        article.source,
        article.pubDate
      ]);
    }

    console.log(`Saved ${articles.length} news articles`);
  } catch (error) {
    console.error('Error saving news articles:', error.message);
    throw error;
  }
}

/**
 * Save predictions to database
 */
async function savePredictions(predictionDate, targetDate, predictions) {
  try {
    const sql = `
      INSERT INTO predictions
      (prediction_date, target_date, prediction_type, symbol, name, rank, reasoning)
      VALUES (?, ?, ?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        rank = VALUES(rank),
        reasoning = VALUES(reasoning)
    `;

    const allPredictions = [
      ...predictions.predictions_up,
      ...predictions.predictions_down
    ];

    for (const pred of allPredictions) {
      await query(sql, [
        predictionDate,
        targetDate,
        pred.type,
        pred.symbol,
        pred.name,
        pred.rank,
        pred.reasoning
      ]);
    }

    console.log(`Saved ${allPredictions.length} predictions`);
  } catch (error) {
    console.error('Error saving predictions:', error.message);
    throw error;
  }
}

/**
 * Score predictions against actual market performance
 */
async function scorePredictions(predictionDate, actualDate) {
  try {
    // Get predictions for the given date
    const predictions = await query(
      'SELECT * FROM predictions WHERE prediction_date = ?',
      [predictionDate]
    );

    if (predictions.length === 0) {
      console.log('No predictions found for date:', predictionDate);
      return {
        date: actualDate,
        total_predictions: 0,
        correct_predictions: 0,
        accuracy_percent: 0
      };
    }

    let correctCount = 0;
    let totalCount = predictions.length;

    for (const prediction of predictions) {
      // Get actual market data for that symbol
      const actualData = await getMarketData(actualDate, prediction.symbol);

      if (actualData && actualData.change_percent !== null) {
        const predictedUp = prediction.prediction_type === 'UP';
        const actualUp = actualData.change_percent > 0;
        const correct = predictedUp === actualUp;

        if (correct) correctCount++;

        // Save individual prediction score
        await savePredictionScore(
          prediction.id,
          actualData.change_percent,
          correct,
          actualDate
        );
      }
    }

    const accuracy = totalCount > 0 ? (correctCount / totalCount) * 100 : 0;

    // Save daily score
    await saveDailyScore(actualDate, totalCount, correctCount, accuracy);

    return {
      date: actualDate,
      total_predictions: totalCount,
      correct_predictions: correctCount,
      accuracy_percent: accuracy
    };
  } catch (error) {
    console.error('Error scoring predictions:', error.message);
    throw error;
  }
}

/**
 * Get market data for a specific date and symbol
 */
async function getMarketData(date, symbol) {
  try {
    const results = await query(
      'SELECT * FROM market_data WHERE date = ? AND symbol = ?',
      [date, symbol]
    );
    return results.length > 0 ? results[0] : null;
  } catch (error) {
    console.error('Error getting market data:', error.message);
    return null;
  }
}

/**
 * Save individual prediction score
 */
async function savePredictionScore(predictionId, actualChange, correct, scoreDate) {
  try {
    await query(
      `INSERT INTO prediction_scores
       (prediction_id, actual_change_percent, correct, score_date)
       VALUES (?, ?, ?, ?)`,
      [predictionId, actualChange, correct ? 1 : 0, scoreDate]
    );
  } catch (error) {
    console.error('Error saving prediction score:', error.message);
    throw error;
  }
}

/**
 * Save daily overall score
 */
async function saveDailyScore(date, total, correct, accuracy) {
  try {
    await query(
      `INSERT INTO daily_scores
       (date, total_predictions, correct_predictions, accuracy_percent)
       VALUES (?, ?, ?, ?)
       ON DUPLICATE KEY UPDATE
         total_predictions = VALUES(total_predictions),
         correct_predictions = VALUES(correct_predictions),
         accuracy_percent = VALUES(accuracy_percent)`,
      [date, total, correct, accuracy]
    );
  } catch (error) {
    console.error('Error saving daily score:', error.message);
    throw error;
  }
}

/**
 * Get historical scores
 */
async function getHistoricalScores(limit = 30) {
  try {
    const results = await query(
      'SELECT * FROM daily_scores ORDER BY date DESC LIMIT ?',
      [limit]
    );
    return results;
  } catch (error) {
    console.error('Error getting historical scores:', error.message);
    return [];
  }
}

/**
 * Get latest predictions with scores
 */
async function getLatestPredictionsWithScores() {
  try {
    const sql = `
      SELECT
        p.*,
        ps.actual_change_percent,
        ps.correct,
        ps.score_date
      FROM predictions p
      LEFT JOIN prediction_scores ps ON p.id = ps.prediction_id
      WHERE p.prediction_date = (SELECT MAX(prediction_date) FROM predictions)
      ORDER BY p.prediction_type, p.rank
    `;

    const results = await query(sql);
    return results;
  } catch (error) {
    console.error('Error getting latest predictions:', error.message);
    return [];
  }
}

module.exports = {
  saveMarketData,
  saveNewsArticles,
  savePredictions,
  scorePredictions,
  getMarketData,
  getHistoricalScores,
  getLatestPredictionsWithScores
};
