const axios = require('axios');

// European market indices
const EUROPEAN_INDICES = [
  { symbol: '^FTSE', name: 'FTSE 100', country: 'UK' },
  { symbol: '^GDAXI', name: 'DAX', country: 'Germany' },
  { symbol: '^FCHI', name: 'CAC 40', country: 'France' },
  { symbol: '^IBEX', name: 'IBEX 35', country: 'Spain' },
  { symbol: '^AEX', name: 'AEX', country: 'Netherlands' },
  { symbol: '^SSMI', name: 'SMI', country: 'Switzerland' },
  { symbol: 'FTSEMIB.MI', name: 'FTSE MIB', country: 'Italy' },
  { symbol: '^OMXC25', name: 'OMX Copenhagen 25', country: 'Denmark' },
  { symbol: '^OMXHPI', name: 'OMX Helsinki', country: 'Finland' },
  { symbol: '^OMXSPI', name: 'OMX Stockholm', country: 'Sweden' },
  { symbol: '^OSEAX', name: 'Oslo Stock Exchange', country: 'Norway' },
  { symbol: '^ISEQ', name: 'ISEQ Overall', country: 'Ireland' },
  { symbol: '^BFX', name: 'BEL 20', country: 'Belgium' },
  { symbol: '^PSI20', name: 'PSI 20', country: 'Portugal' },
  { symbol: '^ATX', name: 'ATX', country: 'Austria' }
];

/**
 * Fetch current market data for a single symbol using Yahoo Finance API
 */
async function fetchMarketData(symbol) {
  try {
    const url = `https://query1.finance.yahoo.com/v8/finance/chart/${encodeURIComponent(symbol)}`;
    const response = await axios.get(url, {
      params: {
        interval: '1d',
        range: '5d'
      },
      headers: {
        'User-Agent': 'Mozilla/5.0'
      }
    });

    const result = response.data.chart.result[0];
    const quote = result.indicators.quote[0];
    const timestamps = result.timestamp;

    if (!timestamps || timestamps.length === 0) {
      throw new Error(`No data available for ${symbol}`);
    }

    // Get the last two days of data for comparison
    const lastIdx = timestamps.length - 1;
    const prevIdx = lastIdx - 1;

    const currentClose = quote.close[lastIdx];
    const previousClose = prevIdx >= 0 ? quote.close[prevIdx] : currentClose;
    const changePercent = ((currentClose - previousClose) / previousClose) * 100;

    return {
      symbol: symbol,
      timestamp: timestamps[lastIdx],
      open: quote.open[lastIdx],
      close: currentClose,
      high: quote.high[lastIdx],
      low: quote.low[lastIdx],
      volume: quote.volume[lastIdx],
      previousClose: previousClose,
      changePercent: changePercent
    };
  } catch (error) {
    console.error(`Error fetching data for ${symbol}:`, error.message);
    return null;
  }
}

/**
 * Fetch market data for all European indices
 */
async function fetchAllEuropeanIndices() {
  const results = [];

  for (const index of EUROPEAN_INDICES) {
    const data = await fetchMarketData(index.symbol);
    if (data) {
      results.push({
        ...data,
        name: index.name,
        country: index.country
      });
    }
    // Add small delay to avoid rate limiting
    await new Promise(resolve => setTimeout(resolve, 500));
  }

  return results;
}

/**
 * Get historical data for comparison
 */
async function fetchHistoricalData(symbol, days = 30) {
  try {
    const period1 = Math.floor(Date.now() / 1000) - (days * 24 * 60 * 60);
    const period2 = Math.floor(Date.now() / 1000);

    const url = `https://query1.finance.yahoo.com/v8/finance/chart/${encodeURIComponent(symbol)}`;
    const response = await axios.get(url, {
      params: {
        period1,
        period2,
        interval: '1d'
      },
      headers: {
        'User-Agent': 'Mozilla/5.0'
      }
    });

    const result = response.data.chart.result[0];
    const quote = result.indicators.quote[0];
    const timestamps = result.timestamp;

    const historical = [];
    for (let i = 0; i < timestamps.length; i++) {
      historical.push({
        date: new Date(timestamps[i] * 1000).toISOString().split('T')[0],
        open: quote.open[i],
        close: quote.close[i],
        high: quote.high[i],
        low: quote.low[i],
        volume: quote.volume[i]
      });
    }

    return historical;
  } catch (error) {
    console.error(`Error fetching historical data for ${symbol}:`, error.message);
    return [];
  }
}

module.exports = {
  EUROPEAN_INDICES,
  fetchMarketData,
  fetchAllEuropeanIndices,
  fetchHistoricalData
};
